<?php

/**
 * League.Uri (https://uri.thephpleague.com/components/2.0/)
 *
 * @package    League\Uri
 * @subpackage League\Uri\Components
 * @author     Ignace Nyamagana Butera <nyamsprod@gmail.com>
 * @link       https://github.com/thephpleague/uri-components
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
declare (strict_types=1);
namespace GFPDF_Vendor\League\Uri\Components;

use GFPDF_Vendor\League\Uri\Contracts\PathInterface;
use GFPDF_Vendor\League\Uri\Contracts\UriComponentInterface;
use GFPDF_Vendor\League\Uri\Contracts\UriInterface;
use Psr\Http\Message\UriInterface as Psr7UriInterface;
use TypeError;
use function array_pop;
use function array_reduce;
use function end;
use function explode;
use function gettype;
use function implode;
use function is_object;
use function is_string;
use function method_exists;
use function sprintf;
use function strpos;
use function substr;
final class Path extends \GFPDF_Vendor\League\Uri\Components\Component implements \GFPDF_Vendor\League\Uri\Contracts\PathInterface
{
    private const SEPARATOR = '/';
    private const DOT_SEGMENTS = ['.' => 1, '..' => 1];
    private const REGEXP_PATH_ENCODING = '/
        (?:[^A-Za-z0-9_\\-\\.\\!\\$&\'\\(\\)\\*\\+,;\\=%\\:\\/@]+|
        %(?![A-Fa-f0-9]{2}))
    /x';
    /**
     * @var string
     */
    private $path;
    /**
     * {@inheritDoc}
     */
    public static function __set_state(array $properties) : self
    {
        return new self($properties['path']);
    }
    /**
     * New instance.
     *
     * @param mixed|string $path
     */
    public function __construct($path = '')
    {
        $this->path = $this->validate($path);
    }
    /**
     * Validate the component content.
     *
     * @param mixed $path the path content
     *
     * @throws TypeError if the component is no valid
     */
    private function validate($path) : string
    {
        $path = $this->validateComponent($path);
        if (null !== $path) {
            return $path;
        }
        throw new \TypeError('The path can not be null.');
    }
    /**
     * Returns a new instance from an string or a stringable object.
     *
     * @param string|object $path
     */
    public static function createFromString($path = '') : self
    {
        if (\is_object($path) && \method_exists($path, '__toString')) {
            return new self((string) $path);
        }
        if (!\is_string($path)) {
            throw new \TypeError(\sprintf('The path must be a string or a stringable object value, `%s` given', \gettype($path)));
        }
        return new self($path);
    }
    /**
     * Create a new instance from a URI object.
     *
     * @param mixed $uri an URI object
     *
     * @throws TypeError If the URI object is not supported
     */
    public static function createFromUri($uri) : self
    {
        if (!$uri instanceof \GFPDF_Vendor\League\Uri\Contracts\UriInterface && !$uri instanceof \Psr\Http\Message\UriInterface) {
            throw new \TypeError(\sprintf('The object must implement the `%s` or the `%s` interface.', \Psr\Http\Message\UriInterface::class, \GFPDF_Vendor\League\Uri\Contracts\UriInterface::class));
        }
        $path = $uri->getPath();
        $authority = $uri->getAuthority();
        if (null === $authority || '' === $authority) {
            return new self($path);
        }
        if ('' === $path || '/' === $path[0]) {
            return new self($path);
        }
        return new self('/' . $path);
    }
    /**
     * {@inheritDoc}
     */
    public function getContent() : ?string
    {
        return $this->encodeComponent($this->path, self::REGEXP_PATH_ENCODING);
    }
    /**
     * {@inheritDoc}
     */
    public function getUriComponent() : string
    {
        return (string) $this->getContent();
    }
    /**
     * Returns the decoded path.
     */
    public function decoded() : string
    {
        return $this->path;
    }
    /**
     * {@inheritDoc}
     */
    public function isAbsolute() : bool
    {
        return self::SEPARATOR === ($this->path[0] ?? '');
    }
    /**
     * Returns whether or not the path has a trailing delimiter.
     */
    public function hasTrailingSlash() : bool
    {
        return '' !== $this->path && self::SEPARATOR === \substr($this->path, -1);
    }
    /**
     * {@inheritDoc}
     */
    public function withContent($content) : \GFPDF_Vendor\League\Uri\Contracts\UriComponentInterface
    {
        $content = self::filterComponent($content);
        if ($content === $this->getContent()) {
            return $this;
        }
        return new self($content);
    }
    /**
     * {@inheritDoc}
     */
    public function withoutDotSegments() : \GFPDF_Vendor\League\Uri\Contracts\PathInterface
    {
        $current = $this->__toString();
        if (\false === \strpos($current, '.')) {
            return $this;
        }
        $input = \explode(self::SEPARATOR, $current);
        $new = \implode(self::SEPARATOR, \array_reduce($input, [$this, 'filterDotSegments'], []));
        if (isset(self::DOT_SEGMENTS[\end($input)])) {
            $new .= self::SEPARATOR;
        }
        return new self($new);
    }
    /**
     * Filter Dot segment according to RFC3986.
     *
     * @see http://tools.ietf.org/html/rfc3986#section-5.2.4
     *
     * @return string[]
     */
    private function filterDotSegments(array $carry, string $segment) : array
    {
        if ('..' === $segment) {
            \array_pop($carry);
            return $carry;
        }
        if (!isset(self::DOT_SEGMENTS[$segment])) {
            $carry[] = $segment;
        }
        return $carry;
    }
    /**
     * Returns an instance with a trailing slash.
     *
     * This method MUST retain the state of the current instance, and return
     * an instance that contains the path component with a trailing slash
     */
    public function withTrailingSlash() : \GFPDF_Vendor\League\Uri\Contracts\PathInterface
    {
        return $this->hasTrailingSlash() ? $this : new self($this->__toString() . self::SEPARATOR);
    }
    /**
     * Returns an instance without a trailing slash.
     *
     * This method MUST retain the state of the current instance, and return
     * an instance that contains the path component without a trailing slash
     */
    public function withoutTrailingSlash() : \GFPDF_Vendor\League\Uri\Contracts\PathInterface
    {
        return !$this->hasTrailingSlash() ? $this : new self(\substr($this->__toString(), 0, -1));
    }
    /**
     * {@inheritDoc}
     */
    public function withLeadingSlash() : \GFPDF_Vendor\League\Uri\Contracts\PathInterface
    {
        return $this->isAbsolute() ? $this : new self(self::SEPARATOR . $this->__toString());
    }
    /**
     * {@inheritDoc}
     */
    public function withoutLeadingSlash() : \GFPDF_Vendor\League\Uri\Contracts\PathInterface
    {
        return !$this->isAbsolute() ? $this : new self(\substr($this->__toString(), 1));
    }
}
