<?php

/**
 * League.Uri (https://uri.thephpleague.com/components/2.0/)
 *
 * @package    League\Uri
 * @subpackage League\Uri\Components
 * @author     Ignace Nyamagana Butera <nyamsprod@gmail.com>
 * @link       https://github.com/thephpleague/uri-components
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */
declare (strict_types=1);
namespace GFPDF_Vendor\League\Uri\Components;

use GFPDF_Vendor\League\Uri\Contracts\UriComponentInterface;
use GFPDF_Vendor\League\Uri\Contracts\UriInterface;
use GFPDF_Vendor\League\Uri\Exceptions\SyntaxError;
use Psr\Http\Message\UriInterface as Psr7UriInterface;
use TypeError;
use function preg_match;
use function sprintf;
use function strtolower;
final class Scheme extends \GFPDF_Vendor\League\Uri\Components\Component
{
    private const REGEXP_SCHEME = ',^[a-z]([-a-z0-9+.]+)?$,i';
    /**
     * @var string|null
     */
    private $scheme;
    /**
     * New instance.
     *
     * @param mixed|null $scheme
     */
    public function __construct($scheme = null)
    {
        $this->scheme = $this->validate($scheme);
    }
    /**
     * Validate a scheme.
     *
     * @param mixed|null $scheme
     *
     * @throws SyntaxError if the scheme is invalid
     * @throws TypeError   if the scheme type is not supported
     */
    private function validate($scheme) : ?string
    {
        $scheme = self::filterComponent($scheme);
        if (null === $scheme) {
            return $scheme;
        }
        if (1 === \preg_match(self::REGEXP_SCHEME, $scheme)) {
            return \strtolower($scheme);
        }
        throw new \GFPDF_Vendor\League\Uri\Exceptions\SyntaxError(\sprintf("The scheme '%s' is invalid.", $scheme));
    }
    /**
     * {@inheritDoc}
     */
    public static function __set_state(array $properties) : self
    {
        return new self($properties['scheme']);
    }
    /**
     * Create a new instance from a URI object.
     *
     * @param mixed $uri an URI object
     *
     * @throws TypeError If the URI object is not supported
     */
    public static function createFromUri($uri) : self
    {
        if ($uri instanceof \GFPDF_Vendor\League\Uri\Contracts\UriInterface) {
            return new self($uri->getScheme());
        }
        if (!$uri instanceof \Psr\Http\Message\UriInterface) {
            throw new \TypeError(\sprintf('The object must implement the `%s` or the `%s` interface.', \Psr\Http\Message\UriInterface::class, \GFPDF_Vendor\League\Uri\Contracts\UriInterface::class));
        }
        $component = $uri->getScheme();
        if ('' === $component) {
            return new self();
        }
        return new self($component);
    }
    /**
     * {@inheritDoc}
     */
    public function getContent() : ?string
    {
        return $this->scheme;
    }
    /**
     * {@inheritDoc}
     */
    public function getUriComponent() : string
    {
        return $this->getContent() . (null === $this->scheme ? '' : ':');
    }
    /**
     * {@inheritDoc}
     */
    public function withContent($content) : \GFPDF_Vendor\League\Uri\Contracts\UriComponentInterface
    {
        $content = $this->validate(self::filterComponent($content));
        if ($content === $this->scheme) {
            return $this;
        }
        return new self($content);
    }
}
