<?php
/**
 *  Object that handle importing content from:
 *  - Surfer
 *  - Google Docs
 *
 * @package SurferSEO
 * @link https://surferseo.com
 */

namespace SurferSEO\Surfer;

use SurferSEO\Surferseo;
use DOMDocument;

/**
 * Object that imports data from different sources into WordPress.
 */
class Content_Importer {

	/**
	 * Title of the imported post.
	 *
	 * @var string
	 */
	protected $title;

	/**
	 * Basic construct.
	 */
	public function __construct() {

		add_action( 'rest_api_init', array( $this, 'register_endpoints' ) );
	}

	/**
	 * Register endpoints for Surfer API.
	 *
	 * @return void
	 */
	public function register_endpoints() {
	}

	/**
	 * Save imported data in database.
	 *
	 * @param string $content - post content.
	 * @param array  $args    - array of optional params.
	 * @return int|WP_Error
	 */
	public function save_data_into_database( $content, $args = array() ) {

		$this->title = wp_strip_all_tags( $this->get_title_from_content( $content ) );
		$content     = $this->parse_content( $content );

		$data = array(
			'post_title'   => $this->title,
			'post_content' => $content,
		);

		if ( isset( $args['post_id'] ) && $args['post_id'] > 0 ) {

			$post_id    = $args['post_id'];
			$data['ID'] = $post_id;
			$post       = (array) get_post( $post_id );

			// WordPress set current date as default and we do not want to change publication date.
			if ( 'published' === $post['post_status'] ) {
				$data['post_date'] = $post['post_date'];
			}

			// Create copy of the post as a backup.
			unset( $post['ID'] );
			$post['post_status'] = 'surfer-backup';
			wp_insert_post( $post );

			$post_id = wp_update_post( $data );
		} else {
			$this->resolve_post_author( $args, $data );
			$this->resolve_post_status( $args, $data );
			$this->resolve_post_date( $args, $data );
			$this->resolve_post_permalink( $args, $data );
			$this->resolve_post_category( $args, $data );
			$this->resolve_post_tags( $args, $data );
			$this->resolve_post_meta_details( $args, $data );

			$post_id = wp_insert_post( $data );
		}

		if ( ! is_wp_error( $post_id ) && isset( $args['draft_id'] ) ) {
			update_post_meta( $post_id, 'surfer_draft_id', $args['draft_id'] );
			update_post_meta( $post_id, 'surfer_keywords', $args['keywords'] );
			update_post_meta( $post_id, 'surfer_location', $args['location'] );
			update_post_meta( $post_id, 'surfer_scrape_ready', true );
			update_post_meta( $post_id, 'surfer_last_post_update', round( microtime( true ) * 1000 ) );
		}

		return $post_id;
	}

	/**
	 * Fill $data array with proper attribute for post_author or leave empty to fill default.
	 *
	 * @param array $args - array of arguments pasted to request.
	 * @param array $data - pointer to array where we store data to put into post.
	 * @return void
	 */
	private function resolve_post_author( $args, &$data ) {

		if ( isset( $args['post_author'] ) && null !== $args['post_author'] ) {
			if ( is_numeric( $args['post_author'] ) && $args['post_author'] > 0 ) {
				$data['post_author'] = $args['post_author'];
			} else {
				$data['post_author'] = $this->get_user_id_by_login( $args['post_author'] );
			}
		} else {
			$default = Surfer()->get_surfer_settings()->get_option( 'content-importer', 'default_post_author', false );

			if ( false !== $default ) {
				$data['post_author'] = $default;
			}
		}
	}

	/**
	 * Fill $data array with proper attribute for post_status or leave empty to fill default.
	 *
	 * @param array $args - array of arguments pasted to request.
	 * @param array $data - pointer to array where we store data to put into post.
	 * @return void
	 */
	private function resolve_post_status( $args, &$data ) {

		$allowed_statuses = array(
			'published',
			'draft',
		);

		if ( isset( $args['post_status'] ) && in_array( $args['post_status'], $allowed_statuses, true ) ) {
			$data['post_status'] = $args['post_status'];
		} else {
			$default = Surferseo::get_instance()->get_surfer_settings()->get_option( 'content-importer', 'default_post_status', false );

			if ( false !== $default ) {
				$data['post_status'] = $default;
			}
		}
	}

	/**
	 * Fill $data array with proper attribute for post_date or leave empty to fill default.
	 *
	 * @param array $args - array of arguments pasted to request.
	 * @param array $data - pointer to array where we store data to put into post.
	 * @return void
	 */
	private function resolve_post_date( $args, &$data ) {

		if ( isset( $args['post_date'] ) && is_date( $args['post_date'] ) ) {
			$data['post_date'] = $args['post_date'];
		}
	}

	/**
	 * Fill $data array with proper attribute for post_name or leave empty to fill default.
	 *
	 * @param array $args - array of arguments pasted to request.
	 * @param array $data - pointer to array where we store data to put into post.
	 * @return void
	 */
	private function resolve_post_permalink( $args, &$data ) {

		if ( isset( $args['post_name'] ) && '' !== $args['post_name'] ) {
			$data['post_name'] = $args['post_name'];
		}
	}

	/**
	 * Fill $data array with proper attribute for post_category or leave empty to fill default.
	 *
	 * @param array $args - array of arguments pasted to request.
	 * @param array $data - pointer to array where we store data to put into post.
	 * @return void
	 */
	private function resolve_post_category( $args, &$data ) {

		if ( isset( $args['post_category'] ) && '' !== $args['post_category'] ) {
			$data['post_category'] = $args['post_category'];
		} else {
			$default = Surferseo::get_instance()->get_surfer_settings()->get_option( 'content-importer', 'default_category', false );

			if ( false !== $default ) {
				$data['post_category'] = array( $default );
			}
		}
	}

	/**
	 * Fill $data array with proper attribute for tags_input or leave empty to fill default.
	 *
	 * @param array $args - array of arguments pasted to request.
	 * @param array $data - pointer to array where we store data to put into post.
	 * @return void
	 */
	private function resolve_post_tags( $args, &$data ) {

		if ( isset( $args['tags_input'] ) && '' !== $args['tags_input'] ) {
			$data['tags_input'] = $args['tags_input'];
		} else {
			$default = Surferseo::get_instance()->get_surfer_settings()->get_option( 'content-importer', 'default_tags', false );

			if ( false !== $default ) {
				$data['tags_input'] = $default;
			}
		}
	}

	/**
	 * Fill the meta_title and meta_description if any SEO plugin is active.
	 *
	 * @param array $args - array of arguments pasted to request.
	 * @param array $data - pointer to array where we store data to put into post.
	 * @return void
	 */
	private function resolve_post_meta_details( $args, &$data ) {

		$seo_plugin_is_active = false;

		if ( ! isset( $data['meta_input'] ) ) {
			$data['meta_input'] = array();
		}

		// Yoast SEO is active.
		if ( $this->check_if_plugins_is_active( 'wordpress-seo/wp-seo.php' ) ) {

			if ( isset( $args['meta_title'] ) && '' !== $args['meta_title'] ) {
				$data['meta_input']['_yoast_wpseo_title'] = $args['meta_title'];
			}

			if ( isset( $args['meta_description'] ) && '' !== $args['meta_description'] ) {
				$data['meta_input']['_yoast_wpseo_metadesc'] = $args['meta_description'];
			}

			$seo_plugin_is_active = true;
		}

		// All in One SEO is active.
		if ( $this->check_if_plugins_is_active( 'all-in-one-seo-pack/all_in_one_seo_pack.php' ) ) {

			if ( isset( $args['meta_title'] ) && '' !== $args['meta_title'] ) {
				$data['meta_input']['_aioseo_title'] = $args['meta_title'];
			}

			if ( isset( $args['meta_description'] ) && '' !== $args['meta_description'] ) {
				$data['meta_input']['_aioseo_description'] = $args['meta_description'];
			}

			$seo_plugin_is_active = true;
		}

		// Rank Math SEO.
		if ( $this->check_if_plugins_is_active( 'seo-by-rank-math/rank-math.php' ) ) {

			if ( isset( $args['meta_title'] ) && '' !== $args['meta_title'] ) {
				$data['meta_input']['rank_math_title'] = $args['meta_title'];
			}

			if ( isset( $args['meta_description'] ) && '' !== $args['meta_description'] ) {
				$data['meta_input']['rank_math_description'] = $args['meta_description'];
			}

			$seo_plugin_is_active = true;
		}

		// Save in Surfer Meta to display.
		if ( ! $seo_plugin_is_active ) {

			if ( isset( $args['meta_title'] ) && '' !== $args['meta_title'] ) {
				$data['meta_input']['_surferseo_title'] = $args['meta_title'];
			}

			if ( isset( $args['meta_description'] ) && '' !== $args['meta_description'] ) {
				$data['meta_input']['_surferseo_description'] = $args['meta_description'];
			}
		}
	}

	/**
	 * Extract h1 from content, to use it as post title.
	 *
	 * @param string $content - Content from Surfer.
	 * @return string
	 */
	private function get_title_from_content( $content ) {

		preg_match( '~<h1[^>]*>(.*?)</h1>~i', $content, $match );
		$title = $match[1];

		return $title;
	}

	/**
	 * Parse content from Surfer into one of used editors.
	 *
	 * @param string $content - Content from Surfer.
	 * @return string
	 */
	private function parse_content( $content ) {

		set_time_limit( 120 );

		$force_editor   = Surfer()->get_surfer_settings()->get_option( 'content-importer', 'default_content_editor', false );
		$parsed_content = '';

		if ( false !== $force_editor ) {
			$parsed_content = $this->resolve_manual_parser_selection( $force_editor, $content );
		} else {
			$parsed_content = $this->resolve_automatic_parser_selection( $content );

		}

		return apply_filters( 'surfer_import_content_parsing', $parsed_content );
	}

	/**
	 * Select content parser based on user settings.
	 *
	 * @param string $editor  - Editor selected by user.
	 * @param string $content - Content from Surfer.
	 */
	private function resolve_manual_parser_selection( $editor, $content ) {
		if ( 'classic' === $editor ) {
			return $this->parse_content_to_classic_editor( $content );
		}

		if ( 'gutenberg' === $editor ) {
			return $this->parse_content_to_gutenberg( $content );
		}

		return $this->resolve_automatic_parser_selection( $content );
	}

	/**
	 * Automatically select parser based on WP version and active plugins.
	 *
	 * @param string $content - Content from Surfer.
	 */
	private function resolve_automatic_parser_selection( $content ) {
		// To classic editor.
		if ( $this->if_user_is_using_classic_editor() ) {
			return $this->parse_content_to_classic_editor( $content );
		}

		// To Gutenberg (default).
		return $this->parse_content_to_gutenberg( $content );
	}

	/**
	 * Checks if user is using Classic editor:
	 *
	 * There are two cases:
	 * - WordPress version < 5.0 (before Gutenberg), without Gutenberg plugin.
	 * - WordPress version >= 5.0 (builtin Gutenberg), with Disable Gutenberg or Classing Editor plugin.
	 *
	 * @return bool
	 */
	private function if_user_is_using_classic_editor() {

		// ( WP < 5.0 && ! Gutenberg Plugin ) || ( WP > 5.0 && Gutendber Plugin )
		$wp_version                     = get_bloginfo( 'version' );
		$if_gutenberg_is_active         = $this->check_if_plugins_is_active( 'gutenberg/gutenberg.php' );
		$if_disable_gutenberg_is_active = $this->check_if_plugins_is_active( 'disable-gutenberg/disable-gutenberg.php' );
		$if_classic_editor_is_active    = $this->check_if_plugins_is_active( 'classic-editor/classic-editor.php' );

		if ( version_compare( $wp_version, '5.0', '>=' ) && ( $if_disable_gutenberg_is_active || $if_classic_editor_is_active ) ) {
			return true;
		}

		if ( version_compare( $wp_version, '5.0', '<' ) && ! $if_gutenberg_is_active ) {
			return true;
		}

		return false;
	}

	/**
	 * Parse content from Surfer to Classic Editor.
	 *
	 * @param string $content - Content from Surfer.
	 * @return string
	 */
	private function parse_content_to_classic_editor( $content ) {

		$content = $this->parse_img_for_classic_editor( $content );
		return $content;
	}

	/**
	 * Parse images for classic editor.
	 *
	 * @param string $content - whole content.
	 * @return string content where <img> URLs are corrected to media library.
	 */
	private function parse_img_for_classic_editor( $content ) {

		$doc = new DOMDocument();
		$doc->loadHTML( $content );

		$h1s = $doc->getElementsByTagName( 'h1' );

		foreach ( $h1s as $h1 ) {
			$h1_text = $this->get_inner_html( $h1 );
			if ( wp_strip_all_tags( $h1_text ) === $this->title ) {
				$content = str_replace( $h1, '', $content );
			}
		}

		$tags = $doc->getElementsByTagName( 'img' );

		foreach ( $tags as $tag ) {
			$image_url = $tag->getAttribute( 'src' );
			$image_alt = $tag->getAttribute( 'alt' );

			$media_library_image_url = $this->download_img_to_media_library( $image_url, $image_alt );

			$content = str_replace( $image_url, $media_library_image_url, $content );
		}

		return $content;
	}

	/**
	 * Parse content from Surfer to Gutenberg Editor.
	 *
	 * @param string $content  - Content from Surfer.
	 * @return string
	 */
	private function parse_content_to_gutenberg( $content ) {

		$content = wp_unslash( $content );

		$doc = new DOMDocument();

		$utf8_fix_prefix = '<html><head><meta http-equiv="content-type" content="text/html; charset=utf-8" /></head><body>';
		$utf8_fix_suffix = '</body></html>';

		$doc->loadHTML( $utf8_fix_prefix . $content . $utf8_fix_suffix, LIBXML_HTML_NODEFDTD | LIBXML_SCHEMA_CREATE );

		$parsed_content = '';

		$this->parse_dom_node( $doc, $parsed_content );

		return $parsed_content;
	}

	/**
	 * Function interates by HTML tags in provided content.
	 *
	 * @param DOMDocument $parent_node - node to parse.
	 * @param string      $content     - reference to content variable, to store Gutenberg output.
	 * @return void
	 */
	private function parse_dom_node( $parent_node, &$content ) {
		// @codingStandardsIgnoreLine
		foreach ( $parent_node->childNodes as $node ) {

			// @codingStandardsIgnoreLine
			$execute_for_child = $this->check_if_execute_recurrence( $node->nodeName );

			$content .= $this->parse_certain_node_type( $node );

			if ( $execute_for_child && $node->hasChildNodes() ) {
				$this->parse_dom_node( $node, $content );
			}
		}
	}

	/**
	 * Function checks if we want to dig deep into content scheme.
	 *
	 * @param string $node_type - name of the node, example: ul, p, h1.
	 * @return bool
	 */
	private function check_if_execute_recurrence( $node_type ) {

		$execute_for_child = true;

		if ( 'ul' === $node_type ) {
			$execute_for_child = false;
		}

		if ( 'ol' === $node_type ) {
			$execute_for_child = false;
		}

		if ( 'blockquote' === $node_type ) {
			$execute_for_child = false;
		}

		return $execute_for_child;
	}

	/**
	 * Function prepares attributes and run correct parser function for certain node type.
	 *
	 * @param DOMElement $node - node to parse.
	 * @return string
	 */
	private function parse_certain_node_type( $node ) {

		$all_attributes = $this->parse_node_attributes( $node );
		$attributes     = $all_attributes['attributes'];

		// @codingStandardsIgnoreLine
		$node_name = $node->nodeName;

		$gutenberg_attribute = '';
		if ( isset( $all_attributes['gutenberg_block_specific'][ $node_name ] ) ) {
			$gutenberg_attribute = $all_attributes['gutenberg_block_specific'][ $node_name ];
		}

		if ( 'p' === $node_name ) {
			return $this->parse_node_p( $node, $attributes, $gutenberg_attribute );
		}

		if ( 0 === strpos( $node_name, 'h' ) && 'html' !== $node_name && 'hr' !== $node_name && 'head' !== $node_name ) {
			return $this->parse_node_h( $node, $attributes, $gutenberg_attribute );
		}

		if ( 'img' === $node_name ) {
			return $this->parse_node_img( $node, $attributes, $gutenberg_attribute );
		}

		if ( 'ul' === $node_name ) {
			return $this->parse_node_ul( $node, $attributes, $gutenberg_attribute );
		}

		if ( 'ol' === $node_name ) {
			return $this->parse_node_ol( $node, $attributes, $gutenberg_attribute );
		}

		if ( 'hr' === $node_name ) {
			return $this->parse_node_hr( $node, $attributes, $gutenberg_attribute );
		}

		if ( 'blockquote' === $node_name ) {
			return $this->parse_node_blockquote( $node, $attributes, $gutenberg_attribute );
		}

		if ( 'pre' === $node_name ) {
			return $this->parse_node_code( $node, $attributes, $gutenberg_attribute );
		}
	}

	/**
	 * Functions prepare attributes for HTML and Gutendber tags.
	 *
	 * @param DOMElement $node - node to parse.
	 * @return array
	 */
	private function parse_node_attributes( $node ) {

		$attributes_array          = array();
		$block_specific_attributes = array();

		if ( $node->hasAttributes() ) {

			// @codingStandardsIgnoreLine
			$node_name  = $node->nodeName;

			foreach ( $node->attributes as $attr ) {

				// @codingStandardsIgnoreLine
				$attr_name  = $attr->nodeName;
				// @codingStandardsIgnoreLine
				$attr_value = $attr->nodeValue;

				if ( 'contenteditable' === $attr_name ) {
					continue;
				}

				$attributes_array[ $attr_name ] = $attr_value;
			}

			if ( in_array( $node_name, array( 'h2', 'h3', 'h4', 'h5', 'h6', 'h7' ), true ) && 'style' === $attr_name ) {
				$this->parse_h_special_attributes( $attr_value, $attributes_array, $block_specific_attributes );
			}

			if ( 'p' === $node_name && 'style' === $attr_name ) {
				$this->parse_p_special_attributes( $attr_value, $attributes_array, $block_specific_attributes );
			}
		}

		return array(
			'attributes'               => $attributes_array,
			'gutenberg_block_specific' => $block_specific_attributes,
		);
	}

	/**
	 * Parse attributes that are specific for the <p> tag.
	 *
	 * @param string $styles_string             - string with all styles from style attribute.
	 * @param array  $attributes_array          - reference to final array of attributes.
	 * @param array  $block_specific_attributes - reference to final array of gutenberg attributes.
	 */
	private function parse_p_special_attributes( $styles_string, &$attributes_array, &$block_specific_attributes ) {

		$styles       = explode( ';', $styles_string );
		$styles_assoc = array();
		foreach ( $styles as $style ) {
			$s                     = explode( ':', $style );
			$styles_assoc[ $s[0] ] = trim( $s[1] );
		}

		if ( key_exists( 'text-align', $styles_assoc ) ) {
			$block_specific_attributes['p'] = '{"align":"' . $styles_assoc['text-align'] . '"} ';

			if ( ! isset( $attributes_array['class'] ) ) {
				$attributes_array['class'] = '';
			}

			$attributes_array['class'] .= ' has-text-align-' . $styles_assoc['text-align'];
		}
	}

	/**
	 * Parse attributes that are specific for the <h2-7> tag.
	 *
	 * @param string $styles_string             - string with all styles from style attribute.
	 * @param array  $attributes_array          - reference to final array of attributes.
	 * @param array  $block_specific_attributes - reference to final array of gutenberg attributes.
	 */
	private function parse_h_special_attributes( $styles_string, &$attributes_array, &$block_specific_attributes ) {

		$styles       = explode( ';', $styles_string );
		$styles_assoc = array();
		foreach ( $styles as $style ) {
			$s                     = explode( ':', $style );
			$styles_assoc[ $s[0] ] = trim( $s[1] );
		}

		if ( key_exists( 'text-align', $styles_assoc ) ) {

			$styles_assoc['text-align'] = str_replace( 'start', 'left', $styles_assoc['text-align'] );
			$styles_assoc['text-align'] = str_replace( 'end', 'right', $styles_assoc['text-align'] );

			$block_specific_attributes['h'] = '"textAlign":"' . $styles_assoc['text-align'] . '"';

			if ( ! isset( $attributes_array['class'] ) ) {
				$attributes_array['class'] = '';
			}

			$attributes_array['class'] .= ' has-text-align-' . $styles_assoc['text-align'];
			unset( $attributes_array['style'] );
		}
	}

	/**
	 * Parses <p> node.
	 *
	 * @param DOMElement $node                - node to parse.
	 * @param array      $attributes          - HTML attributes for the node, example: class="" src="".
	 * @param string     $gutenberg_attribute - Attributes for Gutenberg tag.
	 * @return string
	 */
	private function parse_node_p( $node, $attributes, $gutenberg_attribute ) {

		$attributes = $this->glue_attributes( $attributes );

		$node_content = $this->get_inner_html( $node );
		if ( empty( $node_content ) || '' === $node_content ) {
			return '';
		}

		$attributes = str_replace( 'has-text-align-start', '', $attributes );

		$content  = '<!-- wp:paragraph ' . $gutenberg_attribute . '-->' . PHP_EOL;
		$content .= '<p' . $attributes . '>' . $node_content . '</p>' . PHP_EOL;
		$content .= '<!-- /wp:paragraph -->' . PHP_EOL . PHP_EOL;

		return $content;
	}

	/**
	 * Parses <h1-6> nodes
	 *
	 * @param DOMElement $node                - node to parse.
	 * @param array      $attributes          - HTML attributes for the node, example: class="" src="".
	 * @param string     $gutenberg_attribute - Attributes for Gutenberg tag.
	 * @return string
	 */
	private function parse_node_h( $node, $attributes, $gutenberg_attribute ) {

		$h1_text = $this->get_inner_html( $node );

		// @codingStandardsIgnoreLine
		$node_name = $node->nodeName;

		if ( 'h1' === $node_name && wp_strip_all_tags( $h1_text ) === $this->title ) {
			return '';
		}

		$attributes = $this->glue_attributes( $attributes );

		$header_size = str_replace( 'h', '', $node_name );

		$content  = '<!-- wp:heading {"level":' . $header_size . '} -->' . PHP_EOL;
		$content .= '<h' . $header_size . ' ' . $attributes . '>' . $h1_text . '</h' . $header_size . '>' . PHP_EOL;
		$content .= '<!-- /wp:heading -->' . PHP_EOL . PHP_EOL;

		return $content;
	}

	/**
	 * Parses <img> node.
	 *
	 * @param DOMElement $node                - node to parse.
	 * @param string     $attributes          - HTML attributes for the node, example: class="" src="".
	 * @param string     $gutenberg_attribute - Attributes for Gutenberg tag.
	 * @return string
	 */
	private function parse_node_img( $node, $attributes, $gutenberg_attribute ) {

		$image_url = '';
		$image_alt = '';

		if ( isset( $attributes['src'] ) && ! empty( $attributes['src'] ) ) {
			$image_url = $attributes['src'];
		}

		if ( isset( $attributes['alt'] ) && ! empty( $attributes['alt'] ) ) {
			$image_alt = $attributes['alt'];
		}

		$image_url         = $this->download_img_to_media_library( $image_url, $image_alt );
		$attributes['src'] = $image_url;

		$attributes = $this->glue_attributes( $attributes );

		$content  = '<!-- wp:image -->' . PHP_EOL;
		$content .= '<figure class="wp-block-image">' . PHP_EOL;
		$content .= '<img' . $attributes . ' />' . PHP_EOL;
		$content .= '</figure>' . PHP_EOL;
		$content .= '<!-- /wp:image -->' . PHP_EOL . PHP_EOL;

		return $content;
	}

	/**
	 * Saves image from provided URL into WordPress media library
	 *
	 * @param string $image_url - URL to the image.
	 * @param string $image_alt - Alternative text for the image.
	 * @return string URL to image in media library.
	 */
	private function download_img_to_media_library( $image_url, $image_alt = '' ) {

		$file_name = basename( $image_url );
		$image_id  = $this->find_image_by_name( $file_name );
		if ( 0 === $image_id ) {
			$image_id = $this->upload_images_to_wp( $image_url );
		}

		$this->update_image_alt( $image_id, $image_alt );
		$media_library_image_url = wp_get_attachment_url( $image_id );
		return $media_library_image_url;
	}

	/**
	 * Upload image from Surfer to WordPress and replace src to local one.
	 *
	 * @param string $image_url - URL to the image.
	 * @return int
	 */
	private function upload_images_to_wp( $image_url ) {

		if ( ! isset( $image_url ) || empty( $image_url ) || ! wp_http_validate_url( $image_url ) ) {
			return 0;
		}

		require_once ABSPATH . 'wp-admin/includes/media.php';
		require_once ABSPATH . 'wp-admin/includes/file.php';
		require_once ABSPATH . 'wp-admin/includes/image.php';

		$file_name     = basename( $image_url );
		$tmp_directory = download_url( $image_url );

		$extension = pathinfo( $image_url, PATHINFO_EXTENSION );
		if ( ! isset( $extension ) || empty( $extension ) || '' === $extension ) {
			$headers = get_headers( $image_url );
			foreach ( $headers as $header ) {
				if ( false !== strpos( $header, 'Content-Disposition' ) ) {
					preg_match( '~filename="(.*?)\.(.*?)"~i', $header, $match );
					$file_name .= '.' . $match[2];
					break;
				}
			}
		}

		$file_array = array(
			'tmp_name' => $tmp_directory,
			'name'     => $file_name,
		);

		$attachment_id = media_handle_sideload( $file_array );
		update_post_meta( $attachment_id, 'surfer_file_name', $file_name );
		@unlink( $tmp_directory ); // phpcs:ignore

		return $attachment_id;
	}

	/**
	 * Search for image by name and return it's URL.
	 *
	 * @param string $file_name - name of the file.
	 * @return int
	 */
	private function find_image_by_name( $file_name ) {

		$image_id = 0;

		$file_name = explode( '.', $file_name );
		$file_name = $file_name[0];

		$args = array(
			'post_type'      => 'attachment',
			'name'           => sanitize_title( $file_name ),
			'posts_per_page' => 1,
			'post_status'    => 'inherit',
		);

		$matching_images = get_posts( $args );

		if ( $matching_images ) {
			$image    = array_pop( $matching_images );
			$image_id = $image->ID;
		}

		return $image_id;
	}

	/**
	 * Updates alt param for image.
	 *
	 * @param int    $image_id - ID of the attachment to update.
	 * @param string $image_alt - possible alt attribute for image.
	 * @return void
	 */
	private function update_image_alt( $image_id, $image_alt ) {

		if ( isset( $image_alt ) ) {
			update_post_meta( $image_id, '_wp_attachment_image_alt', trim( $image_alt ) );
		}
	}

	/**
	 * Parses <ul> node.
	 *
	 * @param DOMElement $node                - node to parse.
	 * @param array      $attributes          - HTML attributes for the node, example: class="" src="".
	 * @param string     $gutenberg_attribute - Attributes for Gutenberg tag.
	 * @return string
	 */
	private function parse_node_ul( $node, $attributes, $gutenberg_attribute ) {

		$node_content = $this->get_inner_html( $node );
		if ( empty( $node_content ) || '' === $node_content ) {
			return '';
		}

		$content  = '<!-- wp:list -->' . PHP_EOL;
		$content .= '<ul>' . PHP_EOL;
		$content .= $node_content . PHP_EOL;
		$content .= '</ul>' . PHP_EOL;
		$content .= '<!-- /wp:list -->' . PHP_EOL . PHP_EOL;

		return $content;
	}

	/**
	 * Parses <ol> node.
	 *
	 * @param DOMElement $node                - node to parse.
	 * @param array      $attributes          - HTML attributes for the node, example: class="" src="".
	 * @param string     $gutenberg_attribute - Attributes for Gutenberg tag.
	 * @return string
	 */
	private function parse_node_ol( $node, $attributes, $gutenberg_attribute ) {

		$node_content = $this->get_inner_html( $node );
		if ( empty( $node_content ) || '' === $node_content ) {
			return '';
		}

		$content  = '<!-- wp:list {"ordered":true} -->' . PHP_EOL;
		$content .= '<ol>' . PHP_EOL;
		$content .= $node_content . PHP_EOL;
		$content .= '</ol>' . PHP_EOL;
		$content .= '<!-- /wp:list -->' . PHP_EOL . PHP_EOL;

		return $content;
	}

	/**
	 * Parses <hr> node.
	 *
	 * @param DOMElement $node                - node to parse.
	 * @param array      $attributes          - HTML attributes for the node, example: class="" src="".
	 * @param string     $gutenberg_attribute - Attributes for Gutenberg tag.
	 * @return string
	 */
	private function parse_node_hr( $node, $attributes, $gutenberg_attribute ) {

		$attributes = $this->glue_attributes( $attributes );

		$content  = '<!-- wp:separator -->' . PHP_EOL;
		$content .= '<hr class="wp-block-separator"' . $attributes . ' />' . PHP_EOL;
		$content .= '<!-- /wp:separator -->' . PHP_EOL . PHP_EOL;

		return $content;
	}

	/**
	 * Parses <blockquote> node.
	 *
	 * @param DOMElement $node                - node to parse.
	 * @param array      $attributes          - HTML attributes for the node, example: class="" src="".
	 * @param string     $gutenberg_attribute - Attributes for Gutenberg tag.
	 * @return string
	 */
	private function parse_node_blockquote( $node, $attributes, $gutenberg_attribute ) {

		$node_content = $this->get_inner_html( $node );
		if ( empty( $node_content ) || '' === $node_content ) {
			return '';
		}

		$content  = '<!-- wp:quote -->' . PHP_EOL;
		$content .= '<blockquote class="wp-block-quote">' . $node_content . '</blockquote>' . PHP_EOL;
		$content .= '<!-- /wp:quote -->' . PHP_EOL . PHP_EOL;

		return $content;
	}

	/**
	 * Parses <pre> node.
	 *
	 * @param DOMElement $node                - node to parse.
	 * @param array      $attributes          - HTML attributes for the node, example: class="" src="".
	 * @param string     $gutenberg_attribute - Attributes for Gutenberg tag.
	 * @return string
	 */
	private function parse_node_code( $node, $attributes, $gutenberg_attribute ) {

		$node_content = $this->get_inner_html( $node );
		if ( empty( $node_content ) || '' === $node_content ) {
			return '';
		}

		$content  = '<!-- wp:code -->' . PHP_EOL;
		$content .= '<pre class="wp-block-code">' . PHP_EOL;
		$content .= $node_content . PHP_EOL;
		$content .= '</pre>' . PHP_EOL;
		$content .= '<!-- /wp:code -->' . PHP_EOL . PHP_EOL;

		return $content;
	}

	/**
	 * Turns attributes array into HTML string.
	 *
	 * @param array $attributes_array - array of attributes.
	 * @return string
	 */
	private function glue_attributes( $attributes_array ) {

		$attributes = ' ';

		foreach ( $attributes_array as $key => $value ) {
			$attributes .= $key . '="' . $value . '" ';
		}
		$attributes = rtrim( $attributes );

		return $attributes;
	}

	/**
	 * Returns ID of the user with given name.
	 *
	 * @param string $login - login of the user.
	 * @return int
	 */
	private function get_user_id_by_login( $login = false ) {

		$user_id = 0;
		$user    = get_user_by( 'login', $login );

		if ( false !== $user ) {
			$user_id = get_option( 'surfer_auth_user', 0 );
		}

		return $user_id;
	}

	/**
	 * Extract inner HTML for provided node.
	 *
	 * @param DOMElement $node - node element to parse.
	 * @return string
	 */
	private function get_inner_html( $node ) {
		$inner_html = '';

		// @codingStandardsIgnoreLine
		foreach ( $node->childNodes as $child ) {

			// @codingStandardsIgnoreLine
			$content = $child->ownerDocument->saveXML( $child );

			if ( '<li/>' !== $content ) {
				$inner_html .= $content;
			}
		}

		return $inner_html;
	}

	/**
	 * Checks if plugin is active even if default function is not loaded.
	 *
	 * @param string $plugin - plugin name to check.
	 * @return bool
	 */
	public function check_if_plugins_is_active( $plugin ) {

		if ( ! function_exists( 'is_plugin_active' ) ) {
			return in_array( $plugin, (array) get_option( 'active_plugins', array() ), true );
		} else {
			return is_plugin_active( $plugin );
		}
	}
}
