<?php
/**
 * Class to handle data migration
 *
 * @package SurferSEO
 */

namespace SurferSEO\Surfer\GSC;

/**
 * Class to handle data migration
 */
class Surfer_GSC_Posts_List {

	use Surfer_GSC_Common;

	/**
	 * Object construct.
	 */
	public function __construct() {
		add_action( 'init', array( $this, 'init' ) );
	}

	/**
	 * Init function.
	 */
	public function init() {
		add_action( 'manage_posts_columns', array( $this, 'register_surfer_gsc_data_column' ) );
		add_action( 'manage_posts_custom_column', array( $this, 'render_surfer_gsc_data_column' ) );
	}

	/**
	 * Adds column with GSC data to posts and pages list.
	 *
	 * @param array $columns Columns array.
	 * @return array
	 */
	public function register_surfer_gsc_data_column( $columns ) {
		$post_type = get_post_type();
		if ( ! in_array( $post_type, surfer_return_supported_post_types(), true ) ) {
			return $columns;
		}

		$icon = '<img src="' . Surfer()->get_baseurl() . 'assets/images/surfer_logo_black.svg" alt="surfer_logo" style="max-height: 16px; vertical-align: middle; margin-right: 7px;" />';

		$columns['surfer_gsc_traffic_data'] = $icon . __( 'Surfer', 'surferseo' );
		return $columns;
	}

	/**
	 * Renders column content for GSC data column.
	 *
	 * @param string $column_id Column ID.
	 */
	public function render_surfer_gsc_data_column( $column_id ) {
		if ( 'surfer_gsc_traffic_data' !== $column_id ) {
			return;
		}

		$surfer_gsc_connection = Surfer()->get_surfer_settings()->get_option( 'content-importer', 'surfer_gsc_connection', false );
		if ( ! isset( $surfer_gsc_connection ) || 1 !== intval( $surfer_gsc_connection ) ) {
			echo '<a href="' . esc_attr( admin_url( 'admin.php?page=surfer' ) ) . '">' . esc_html__( 'Add GSC', 'surferseo' ) . '</a>';
		} else {

			$post         = get_post();
			$post_traffic = surfer_get_last_post_traffic_by_id( $post->ID );

			if ( $post_traffic ) {
				$this->render_position_monitor_column_values( $post->ID );
			} elseif ( 'publish' !== $post->post_status ) {
				esc_html_e( 'Publish a post to see data from GSC.', 'surferseo' );
			} else {
				esc_html_e( 'Relax while we\'re gathering your data.', 'surferseo' );
			}
		}
	}

		/**
		 * Renders position monitor column values.
		 *
		 * @param int $post_id Post ID.
		 */
	private function render_position_monitor_column_values( $post_id ) {

		$post_performance = surfer_get_last_post_traffic_by_id( $post_id );

		$last_update_date     = $post_performance['data_gathering_date'];
		$previous_update_date = $this->get_previous_period_date( $post_id );

		$draft_id = get_post_meta( $post_id, 'surfer_draft_id', true );

		$position      = $post_performance['position'];
		$prev_position = $post_performance['position'] - $post_performance['position_change'];

		$clicks      = $post_performance['clicks'];
		$prev_clicks = $post_performance['clicks'] - $post_performance['clicks_change'];

		$impressions      = $post_performance['impressions'];
		$prev_impressions = $post_performance['impressions'] - $post_performance['impressions_change'];

		$position_direction = '';
		if ( $position > $prev_position ) {
			$position_direction = 'down';
		} elseif ( $position < $prev_position ) {
			$position_direction = 'up';
		}

		ob_start();
			require Surfer()->get_basedir() . '/templates/admin/posts-list-gsc-column.php';
		$html = ob_get_clean();

		$additional_allowed_html = array(
			'svg'  => array(
				'xmlns'        => array(),
				'fill'         => array(),
				'viewbox'      => array(),
				'role'         => array(),
				'aria-hidden'  => array(),
				'focusable'    => array(),
				'stroke-width' => array(),
				'stroke'       => array(),
				'class'        => array(),
			),
			'path' => array(
				'd'               => array(),
				'fill'            => array(),
				'stroke-linecap'  => array(),
				'stroke-linejoin' => array(),

			),
			'b'    => array(),
			'br'   => array(),
		);

		echo wp_kses( $html, array_merge( wp_kses_allowed_html( 'post' ), $additional_allowed_html ) );
	}
}
